package org.msh.tb.reports2.variables.concat;

import org.msh.reports.FilterValue;
import org.msh.reports.filters.Filter;
import org.msh.reports.filters.FilterOperation;
import org.msh.reports.filters.ValueHandler;
import org.msh.reports.keys.Key;
import org.msh.reports.query.SQLDefs;
import org.msh.reports.variables.Variable;
import org.msh.utils.Tuple;

import java.util.List;
import java.util.Map;

/**
 * Variable that concatenates the result of other variables, making it possible
 * to arrange variable results in a single row (or column)
 *
 * Created by rmemoria on 11/12/16.
 */
public class ConcatVariable implements Variable {

    private static final String KEY_TOTAL = "$total";

    private List<ConcatVariableInfo> variables;

    /**
     * Optional list of group keys and titles to group variables
     */
    private Map<String, String> groups;

    private String id;


    public ConcatVariable(String id) {
        this.id = id;
        parseVariables();
    }

    /**
     * Parse the variables, its fields and parameters defined in the variable ID
     */
    public void parseVariables() {
        ParsedData data = ConcatVariableParser.parse(this.id);
        variables = data.getVariables();
        groups = data.getGroups();

        int groupCount = 0;

        for (ConcatVariableInfo vinfo: variables) {
            if (vinfo.getVariable().isGrouped()) {
                groupCount++;
            }
        }

        // check if all variables have the same grouping
        if (groupCount > 0 && groupCount < variables.size()) {
            throw new RuntimeException("All variables must have the same grouping configuration");
        }
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public String getLabel() {
        String s = "";
        for (ConcatVariableInfo v: variables) {
            if (!s.isEmpty()) {
                s += " / ";
            }
            s += v.getVariable().getLabel();
        }

        return s;
    }

    /**
     * Return the variable and its iteration based on the iteration of the contact variable
     * @param iteration the iteration
     * @return {@link Tuple} containing the variable and its iteration
     */
    private Tuple<ConcatVariableInfo, Integer> variableByIteration(int iteration) {
        int index = iteration;

        for (ConcatVariableInfo vinfo: variables) {
            int count = vinfo.getVariable().getIteractionCount();
            if (count == 0) {
                count = 1;
            }

            if (index < count) {
                return Tuple.of(vinfo, index);
            }

            index -= count;
        }

        throw new RuntimeException("Invalid iteration: " + iteration);
    }

    @Override
    public void prepareVariableQuery(SQLDefs def, int iteration) {
        Tuple<ConcatVariableInfo, Integer> res = variableByIteration(iteration);
        ConcatVariableInfo vinfo = res.getValue1();
        if (vinfo.getFilters() != null) {
            vinfo.getFilters().entrySet();
            for (Map.Entry<Filter, String> entry:  vinfo.getFilters().entrySet()) {
                Filter filter = entry.getKey();
                FilterValue fv = FilterValue.parseString(filter, entry.getValue());
                filter.prepareFilterQuery(def,
                        fv.getComparator(),
                        new ValueHandler(fv.getValue(), filter.isMultiSelection()));
            }
        }
        Variable var = vinfo.getVariable();
        int varIteration = res.getValue2();

        var.prepareVariableQuery(def, varIteration);
    }

    @Override
    public Key createKey(Object values, int iteration) {
        // get the variable according to the iteration
        Tuple<ConcatVariableInfo, Integer> tuple = variableByIteration(iteration);
        ConcatVariableInfo vinfo = tuple.getValue1();
        Variable var = vinfo.getVariable();

        // create key of the original variable
        Key varKey = var.createKey(values, tuple.getValue2());
        varKey.setIteration(tuple.getValue2());
        varKey.setVariable(var);

        // create tuple to be used as key in the concat variable
        Tuple key = Tuple.of(vinfo, varKey);

        // is total enabled ?
        if (tuple.getValue1().isCalcTotal() && var.isTotalEnabled()) {
            Key keyTotal = Key.of(KEY_TOTAL);
            keyTotal.setVariable(var);
            return Key.asMultiple(key, Tuple.of(vinfo, keyTotal));
        }

        Key k = vinfo.getGroupKey() != null ?
                Key.of(vinfo.getGroupKey(), key) :
                Key.of(key);
        return k;
    }

    @Override
    public String getDisplayText(Key key) {
        Tuple<ConcatVariableInfo, Key> t = (Tuple)key.getValue();
        Key varKey = t.getValue2();

        Tuple<ConcatVariableInfo, Integer> res = variableByIteration(key.getIteration());
        ConcatVariableInfo vinfo = res.getValue1();
        Variable var = vinfo.getVariable();

        // check if there is a custom title assigned to the key
        if (vinfo.getKeyTitles() != null) {
            String k = varKey.isNull() ? "null" : varKey.getValue().toString();
            String title = vinfo.getKeyTitles().get(k);
            if (title != null) {
                return title;
            }
        }

        if (KEY_TOTAL.equals(varKey.getValue())) {
            return "Total";
        }

        return var.getDisplayText(varKey);
    }

    @Override
    public int compareValues(Key key1, Key key2) {
        ConcatVariableInfo vinfo1 = ((Tuple<ConcatVariableInfo, Key>)key1.getValue()).getValue1();
        ConcatVariableInfo vinfo2 = ((Tuple<ConcatVariableInfo, Key>)key2.getValue()).getValue1();

        // the variable key is wrapped inside the key
        Key val1 = ((Tuple<ConcatVariableInfo, Key>)key1.getValue()).getValue2();
        Key val2 = ((Tuple<ConcatVariableInfo, Key>)key2.getValue()).getValue2();

        Tuple<ConcatVariableInfo, Integer> t1 = variableByIteration(key1.getIteration());
        Tuple<ConcatVariableInfo, Integer> t2 = variableByIteration(key2.getIteration());

        // compare the variables
        if (vinfo1 != vinfo2) {
            Integer i1 = variables.indexOf(t1.getValue1());
            Integer i2 = variables.indexOf(t2.getValue1());

            return i1.compareTo(i2);
        }

        Variable v1 = t1.getValue1().getVariable();
        Variable v2 = t2.getValue1().getVariable();

        // check the total column
        if (!val1.isNull() && val1.getValue().equals(val2.getValue())) {
            return 0;
        }

        if (KEY_TOTAL.equals(val1.getValue())) {
            return 1;
        }

        if (KEY_TOTAL.equals(val2.getValue())) {
            return -1;
        }


        return v1.compareValues(val1, val2);
    }

    @Override
    public Object[] getDomain() {
        return null;
    }

    @Override
    public boolean isGrouped() {
        return groups != null && !groups.isEmpty();
    }


    @Override
    public String getGroupDisplayText(Key key) {
        return groups != null && key.getGroup() != null ?
                groups.get(key.getGroup()) :
                key.getGroup().toString();
    }

    @Override
    public int getIteractionCount() {
        int count = 0;
        for (ConcatVariableInfo vinfo: variables) {
            int num = vinfo.getVariable().getIteractionCount();
            if (num == 0) {
                count++;
            } else {
                count += vinfo.getVariable().getIteractionCount();
            }
        }
        return count;
    }

    @Override
    public boolean isTotalEnabled() {
        return false;
    }

    @Override
    public Object getUnitType() {
        return variables.get(0).getVariable().getUnitType();
    }

    @Override
    public String getUnitTypeLabel() {
        return variables.get(0).getVariable().getUnitTypeLabel();
    }

    public Map<String, String> getGroups() {
        return groups;
    }

    public void setGroups(Map<String, String> groups) {
        this.groups = groups;
    }
}
